package net.w_horse.excelpojo.excel;

import java.lang.annotation.Annotation;
import java.util.regex.Pattern;

import net.w_horse.excelpojo.ExcelPOJOBridge;
import net.w_horse.excelpojo.ExcelPOJOException;
import net.w_horse.excelpojo.annotation.ExcelPOJOAnnotationParser;
import net.w_horse.excelpojo.xml.ExcelPOJOXmlParser;
import net.w_horse.excelpojo.xml.tag.Use;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.w3c.dom.Element;

public abstract class AbstractCellSeeker {
	private ExcelPOJOBridge excelPOJOBridge;
	private String use = Use.NONE.getValue();
	private String range;

	public abstract void set(Element element, ExcelPOJOXmlParser parser);
	public abstract void set(Annotation annotation, ExcelPOJOAnnotationParser parser) throws ClassNotFoundException, LinkageError;

	public abstract Object seekCellValue(Sheet sheet, Class<?> requiredType) throws ClassNotFoundException, LinkageError, CellNotFoundException, IllegalArgumentException, ExcelPOJOException;
	protected abstract Offset seekCellPosition(Sheet sheet) throws CellNotFoundException, IllegalArgumentException, ExcelPOJOException;
	protected abstract Offset seekCellPosition(Sheet sheet, Offset basePosition, Offset offset) throws CellNotFoundException, IllegalArgumentException, ExcelPOJOException;
	public abstract void setValue(Sheet sheet, Object value) throws CellNotFoundException, IllegalArgumentException, ExcelPOJOException;
	protected abstract void setValue(Sheet sheet, Offset basePosition, Offset offset, Object value) throws CellNotFoundException;


	public boolean verify() throws IllegalArgumentException, ExcelPOJOException {
		Pattern pattern = Pattern.compile("\\(\\s?-?[0-9]+\\s?,\\s?-?[0-9]+\\s?\\)-\\(\\s?-?[0-9]+\\s?,\\s?-?[0-9]+\\s?\\)");
		if ((getRange() != null) && !getRange().isEmpty() && !pattern.matcher(getRange()).matches()) {
			throw new IllegalArgumentException("There is a mistake in the format of the range specification. \n"
					+ "Format:(startRowIndex, startColIndex)-(endRowIndex, endColIndex)  rowIndex={1,2,3,...}, colIndex={1,2,3,...}");
		}
		return true;
	}

	protected <T>T getCellValue(Sheet sheet, Offset cellPosition, Class<T> requiredType) {
		return getCellValue(sheet, cellPosition, new Offset(), requiredType);
	}

	protected <T>T getCellValue(Sheet sheet, Offset basePosition, Offset offset, Class<T> requiredType) {
		return getCellValue(getCell(sheet, basePosition, offset), requiredType);
	}

	public <T>T getCellValue(Cell cell, Class<T> requiredType) {
		return ExcelUtils.getCellValue(cell, requiredType);
	}
	protected Cell createCell(Sheet sheet, Offset basePosition, Offset offset) {
		Row row = sheet.getRow(basePosition.rowIndex + offset.rowIndex);
		if (row == null) {
			row = sheet.createRow(basePosition.rowIndex + offset.rowIndex);
		}
		Cell cell = row.getCell(basePosition.colIndex + offset.colIndex);
		if (cell == null) {
			cell = row.createCell(basePosition.colIndex + offset.colIndex);
		}
		return cell;
	}

	protected Cell getCell(Sheet sheet, Offset cellPosition) {
		return getCell(sheet, cellPosition, new Offset());
	}
	protected Cell getCell(Sheet sheet, Offset basePosition, Offset offset) {
		if (basePosition == null) return null;

		return createCell(sheet, basePosition, offset);
	}

	protected void setCellValue(Cell cell, Object value) {
		ExcelUtils.setCellValue(cell, value);
	}

	public void setExcelPOJOBridge(ExcelPOJOBridge excelPOJOBridge) {
		this.excelPOJOBridge = excelPOJOBridge;
	}
	public ExcelPOJOBridge getExcelPOJOBridge() {
		return excelPOJOBridge;
	}

	public void setUse(String use) {
		this.use = use;
	}

	public String getUse() {
		return use;
	}

	public void setRange(String range) {
		this.range = range;
	}
	public String getRange() {
		return range;
	}

	protected class Offset {
		public int rowIndex = 0;
		public int colIndex = 0;

		public Offset() {
		}
		public Offset(Offset offset) {
			this.rowIndex = offset.rowIndex;
			this.colIndex = offset.colIndex;
		}
		public Offset(int rowIndex, int colIndex) {
			this.rowIndex = rowIndex;
			this.colIndex = colIndex;
		}

		public Offset add(Offset offset) {
			this.rowIndex += offset.rowIndex;
			this.colIndex += offset.colIndex;

			return this;
		}
		public Offset negate() {
			return new Offset(-this.rowIndex, -this.colIndex);
		}
	}

	protected Offset getRangeStartCell() {
		return convRangeString2Offset(true);
	}
	protected Offset getRangeEndCell() {
		return convRangeString2Offset(false);
	}
	private Offset convRangeString2Offset(boolean start) {
		String[] cellPositions = getRange().split("\\)-\\(");
		String cellPositionString = (start ? cellPositions[0] : cellPositions[1]);
		String[] xy = (start
						? cellPositionString.substring(1).split(",")
						: cellPositionString.substring(0, cellPositionString.length() -1).split(","));
		int rowIndex = Integer.valueOf(xy[0].trim()) -1;
		int colIndex = Integer.valueOf(xy[1].trim()) -1;

		return new Offset(rowIndex, colIndex);
	}
	protected String convertOffset2RangeString(Offset basePosition, Offset range) {
		int startRowIndex = basePosition.rowIndex;
		int startColumnIndex = basePosition.colIndex;
		int endRowIndex = (range.rowIndex < 0 ? -1 : startRowIndex + range.rowIndex);
		int endColumnIndex = (range.colIndex < 0 ? -1 : startColumnIndex + range.colIndex);

		String startCellStr = String.format("(%d,%d)", startRowIndex +1, startColumnIndex +1);
		String endCellStr = String.format("(%d,%d)", endRowIndex +1, endColumnIndex +1);

		return startCellStr + "-" + endCellStr;
	}
	protected int getStartRowIndex(int sheetFirstRowNum, int basePositionRowIndex) {
		int rowIndex = (sheetFirstRowNum > basePositionRowIndex
							? sheetFirstRowNum : basePositionRowIndex);
		if (getRange() != null && !getRange().isEmpty() && (getRangeStartCell().rowIndex > 0)) {
			rowIndex = (rowIndex > getRangeStartCell().rowIndex
							? rowIndex : getRangeStartCell().rowIndex);
		}
		return rowIndex;
	}
	protected int getLimitRowIndex(int sheetLastRowNum) {
		int limitRowIndex = sheetLastRowNum;
		if (getRange() != null && !getRange().isEmpty() && (getRangeEndCell().rowIndex > 0)) {
			limitRowIndex = (limitRowIndex < getRangeEndCell().rowIndex
								? limitRowIndex : getRangeEndCell().rowIndex);
		}
		return limitRowIndex;
	}
	protected int getStartColIndex(int rowFirstCellNum, int basePositionColumnIndex) {
		int colIndex = (rowFirstCellNum > basePositionColumnIndex
							? rowFirstCellNum : basePositionColumnIndex);
		if (getRange() != null && !getRange().isEmpty() && (getRangeStartCell().colIndex > 0)) {
			colIndex = (colIndex > getRangeStartCell().colIndex
							? colIndex : getRangeStartCell().colIndex);
		}
		return colIndex;
	}
	protected int getLimitColumnIndex(int rowLastCellNum) {
		int colIndexLimit = rowLastCellNum;
		if (getRange() != null && !getRange().isEmpty() && (getRangeEndCell().colIndex > 0)) {
			colIndexLimit = (colIndexLimit < getRangeEndCell().colIndex
								? colIndexLimit : getRangeEndCell().colIndex);
		}
		return colIndexLimit;
	}

}
